<?php

namespace App\Http\Controllers;
use App\Models\User;
use App\Models\CustomerType;
use Illuminate\Http\Request;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\DB;
use App\Http\Resources\CustomerType as CustomerTypeCollection;

class CustomerTypeController extends Controller
{
    public function index(Request $request)
    {
        try {
            
            $limit = $request->query('limit', 10);
            $customerType = CustomerType::paginate($limit);
            $customerCollection = CustomerTypeCollection::collection($customerType->items());
            $responseData = [
                'records' => $customerCollection,
                'pagination' => [
                    'current_page' => $customerType->currentPage(),
                    'last_page' => $customerType->lastPage(),
                    'per_page' => $customerType->perPage(),
                    'total' => $customerType->total(),
                ],
            ];

            return response()->json($responseData);
        } catch (QueryException $e) {
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'customer_type' => 'required|string|max:100',
            ]);

            DB::beginTransaction();
            $customerType = CustomerType::create([
                'customer_type' => $request->customer_type,
                'status' => $request->status ?? true, // Default to true if status is not provided.
            ]);
 
            DB::commit();
            return response()->json(['message' => 'Customer Type created successfully', 'record' => new CustomerTypeCollection($customerType)], 201);
        } catch (QueryException $e) {
            DB::rollBack();
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        try{
            $customerType = CustomerType::findOrFail($id);
            return response()->json(['message' => 'Customer Type Fetached successfully', 'record' => new CustomerTypeCollection($customerType)], 200);
        }catch (\Exception $e) {
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
        
    }


    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $validatedData = $request->validate([
                'customer_type' => 'required|string|max:100',
            ]);

            DB::beginTransaction();
    
            $customerType = CustomerType::findOrFail($id);
            $customerType->customer_type = $request->customer_type;
            $customerType->status = $request->status ?? $customerType->status;
            $customerType->save();

            DB::commit();
    
            return response()->json(['message' => 'Customer Type updated successfully', 'record' => new CustomerTypeCollection($customerType)], 200);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['message' => 'Customer Type  not found'], 404);
        } catch (QueryException $e) {
            DB::rollBack();
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            $customerType = CustomerType::findOrFail($id);
            $customerType->delete();
            return response()->json(['message' => 'Customer Type deleted successfully'], 200);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['message' => 'Customer Type not found'], 404);
        } catch (QueryException $e) {
            DB::rollBack();
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
    }
}
