<?php

namespace App\Http\Controllers;

use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\DB;

use App\Models\Price;
use App\Models\PriceData;
use App\Models\PriceCity;

use Illuminate\Http\Request;
use App\Http\Resources\Price as PriceCollection;
use App\Http\Resources\ClosePrice as ClosePriceCollection;

class PriceController extends Controller
{
    public function index(Request $request)
    {
        try {
            $limit = $request->query('limit', 10);

            $query = Price::query();
            // Get all parameters from the request
            $parameters = $request->all();

        // Loop through each parameter and apply filters to the query
            foreach ($parameters as $key => $value) {
                // Make sure the key is not a reserved Laravel parameter
                if ($key !== 'page' && $key !== 'limit' && $key !== 'date_range' && $key !== 'city_id') {
                    // Apply exact match for other fields
                        $query->where($key, $value);
                }
            }
            // Apply date range filter
            if ($request->has('date_range')) {
                $dates = explode(' - ', $request->input('date_range'));
                $startDate = \Carbon\Carbon::createFromFormat('d/m/Y', $dates[0])->startOfDay();
                $endDate = \Carbon\Carbon::createFromFormat('d/m/Y', $dates[1])->endOfDay();
                $query->whereBetween('created_at', [$startDate, $endDate]);
            }

            $prices = $query->orderBy('updated_at', 'desc')->paginate($limit);
            $priceCollection = PriceCollection::collection($prices->items());

            $responseData = [
                'records' => $priceCollection,
                'pagination' => [
                    'current_page' => $prices->currentPage(),
                    'last_page' => $prices->lastPage(),
                    'per_page' => $prices->perPage(),
                    'total' => $prices->total(),
                ],
            ];

            return response()->json($responseData);
        } catch (QueryException $e) {
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
    }

    public function todayPrice(Request $request){

        try {
           // Get the latest price for each product using Eloquent
            $subquery = Price::select('price_product_id', DB::raw('MAX(record_date) as latest_created_at'))
            ->whereIn('price_product_id', function($query) {
                $query->select('id')
                    ->from('price_products')
                    ->where('status', 1);
            })
            ->groupBy('price_product_id');

            // Fetch the full price model by joining with the subquery
            $latestPrices = Price::joinSub($subquery, 'latest', function($join) {
                $join->on('prices.price_product_id', '=', 'latest.price_product_id')
                    ->on('prices.record_date', '=', 'latest.latest_created_at');
            })
            ->orderBy('prices.record_date', 'desc')
            ->get();

            return response()->json(['message' => 'Price Details Fetached successfully', 'records' => PriceCollection::collection($latestPrices)], 200);

        } catch (QueryException $e) {
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }

    }


    public function todayClosePrice(Request $request){

        //try {
           // Get the latest price for each product using Eloquent
            $subquery = Price::select('price_product_id', DB::raw('MAX(record_date) as latest_created_at'))
            ->whereIn('price_product_id', function($query) {
                $query->select('id')
                    ->from('price_products')
                    ->where('status', 1);
            })
            ->groupBy('price_product_id');

            // Fetch the full price model by joining with the subquery
            $latestPrices = Price::joinSub($subquery, 'latest', function($join) {
                $join->on('prices.price_product_id', '=', 'latest.price_product_id')
                    ->on('prices.record_date', '=', 'latest.latest_created_at');
            })
            ->orderBy('prices.record_date', 'desc')
            ->get();

            return response()->json(['message' => 'Price Details Fetached successfully', 'records' => ClosePriceCollection::collection($latestPrices)], 200);

        /*} catch (QueryException $e) {
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }*/

    }




    public function todayPriceByProduct(Request $request, $product_id){

        try {
            // Get the latest price for each product using Eloquent
            $subquery = Price::select('price_product_id', DB::raw('MAX(record_date) as latest_created_at'))
            ->where('price_product_id', $product_id)
            ->groupBy('price_product_id');

            // Fetch the full price model for the single product_id by joining with the subquery
            $latestPrice = Price::joinSub($subquery, 'latest', function ($join) {
                    $join->on('prices.price_product_id', '=', 'latest.price_product_id')
                        ->on('prices.record_date', '=', 'latest.latest_created_at');
                })
                ->where('prices.price_product_id', $product_id)
                ->orderBy('prices.record_date', 'desc')
                ->first();

            return response()->json(['message' => 'Price Details Fetached successfully', 'record' => new PriceCollection($latestPrice)], 200);

        } catch (QueryException $e) {
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }

    }











    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            $price = Price::create([
                'price_product_id' => $request->price_product_id,
                'price_country_id' => $request->price_country_id,
                'status' => $request->status ?? true, 
            ]);

            foreach ($request->city_value as $city) {
                PriceData::create([
                    'price_id' => $price->id,
                    'price_city_id' => $city['price_city_id'],
                    'rate' => $city['rate'],
                ]);
            }
 
            DB::commit();
            return response()->json(['message' => 'Price created successfully', 'record' => new PriceCollection($price)], 201);
        } catch (QueryException $e) {
            DB::rollBack();
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        try{
            $price = Price::findOrFail($id);
            return response()->json(['message' => 'Price Details Fetached successfully', 'record' => new PriceCollection($price)], 200);
        }catch (\Exception $e) {
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
        
    }


    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {

            DB::beginTransaction();

            $price = Price::findOrFail($id); 
            $price->price_product_id = $request->price_product_id;
            $price->price_country_id = $request->price_country_id;
            $price->status = $request->status;
            $price->save();

            $price->price_data()->delete();
            foreach ($request->city_value as $city) {
                PriceData::create([
                    'price_id' => $price->id,
                    'price_city_id' => $city['price_city_id'],
                    'rate' => $city['rate'],
                ]);
            }
            DB::commit();
    
            return response()->json(['message' => 'Price updated successfully', 'record' => new PriceCollection($price)], 200);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['message' => 'Price not found'], 404);
        } catch (QueryException $e) {
            DB::rollBack();
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            $ids = explode(",",$id);
            if (!empty($ids)) {
                // Fetch the product  with the specified IDs
                $prices = Price::whereIn('id', $ids)->get();
                
                // Delete the fetched product groups
                foreach ($prices as $price) {
                    $price->price_data()->delete();
                    $price->delete();
                }
            }
            return response()->json(['message' => 'Price and associated entries deleted successfully'], 200);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['message' => 'Price not found'], 404);
        } catch (QueryException $e) {
            DB::rollBack();
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
    }


    // Other Resources

    public function activePrice()
    {
        try {
            $price = Price::where('status',true)->get();

            return response()->json(['records' => PriceCollection::collection($price)]);
        } catch (QueryException $e) {
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
    }

    public function getpriceProductCity(Request $request)
{
    try {
        $query = Price::query();
        $parameters = $request->all();

        // Loop through each parameter and apply filters to the query
        foreach ($parameters as $key => $value) {
            if ($key !== 'date_range' && $key !== 'price_city_id') {
                $query->where($key, $value);
            }
        }

        // Apply date range filter
        if ($request->has('date_range')) {
            $dates = explode(' - ', $request->input('date_range'));
            $startDate = \Carbon\Carbon::createFromFormat('d/m/Y', $dates[0])->startOfDay();
            $endDate = \Carbon\Carbon::createFromFormat('d/m/Y', $dates[1])->endOfDay();
            $query->whereBetween('record_date', [$startDate, $endDate]);
        }

        // Fetch the prices
        $prices = $query->with('price_data')->orderBy('record_date', 'desc')->get();

        // Filter the price_data by price_city_id if it's provided in the request

        $prices->transform(function ($price) {
            $price->price_data = $price->price_data->filter(function ($data) {
                // Filter out entries where rate is "11"
                return $data->rate !== NULL;
            });
            return $price;
        });



        if ($request->has('price_city_id')) {
            $priceCityId = $request->input('price_city_id');

            // Iterate over each price item and filter its price_data
            $prices->transform(function ($price) use ($priceCityId) {
                $price->price_data = $price->price_data->filter(function ($data) use ($priceCityId) {
                    return $data->price_city_id == $priceCityId;
                });
                return $price;
            });
        }

        $priceCollection = PriceCollection::collection($prices);

        $responseData = [
            'records' => $priceCollection,
        ];

        return response()->json($responseData);
    } catch (QueryException $e) {
        return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
    } catch (\Exception $e) {
        return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
    }
}

    
}