<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\ProductField;
use Illuminate\Http\Request;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\DB;
use App\Http\Resources\Product as ProductCollection;

class ProductController extends Controller
{
    public function index(Request $request)
        {
            try {
                $limit = $request->query('limit', 10);

                $query = Product::query();
                // Get all parameters from the request
                $parameters = $request->all();

            // Loop through each parameter and apply filters to the query
                foreach ($parameters as $key => $value) {
                    // Make sure the key is not a reserved Laravel parameter
                    if ($key !== 'page' && $key !== 'limit') {
                        if (in_array($key, ['product_name'])) {
                            // Apply search functionality using 'like' for name, email, and phone
                            $query->where($key, 'like', "%$value%");
                        } else {
                            // Apply exact match for other fields
                            $query->where($key, $value);
                        }
                    }
                }

                $products = $query->orderBy('updated_at', 'desc')->paginate($limit);
                $productCollection = ProductCollection::collection($products->items());

                $responseData = [
                    'records' => $productCollection,
                    'pagination' => [
                        'current_page' => $products->currentPage(),
                        'last_page' => $products->lastPage(),
                        'per_page' => $products->perPage(),
                        'total' => $products->total(),
                    ],
                ];
                return response()->json($responseData);
                
            } catch (QueryException $e) {
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
    
    
        /**
         * Store a newly created resource in storage.
         */
        public function store(Request $request)
        {
            //try {
                $validatedData = $request->validate([
                    'product_name' => 'required|string|max:100',
                ]);
    
                DB::beginTransaction();
                $product = Product::create([
                    'product_name' => $request->product_name,
                    'product_description' => $request->product_description,
                    'product_image_id' => $request->product_image_id,
                    'product_type' => $request->product_type,
                    'display_box' => $request->display_box,
                    'status' => $request->status ?? true, // Default to true if status is not provided.
                ]);

                foreach ($request->fields as $field) {
                    ProductField::create([
                        'product_id' => $product->id,
                        'field' => $field
                    ]);
                }
     
                DB::commit();
                return response()->json(['message' => 'Product created successfully', 'record' => new ProductCollection($product)], 201);
           /* } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }*/
        }
    
        /**
         * Display the specified resource.
         */
        public function show($id)
        {
            try{
                $product = Product::findOrFail($id);
                return response()->json(['message' => 'Product Details Fetached successfully', 'record' => new ProductCollection($product)], 200);
            }catch (\Exception $e) {
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
            
        }
    
    
        /**
         * Update the specified resource in storage.
         */
        public function update(Request $request, $id)
        {
            try {
                $validatedData = $request->validate([
                    'product_name' => 'required|string|max:100',
                ]);
    
                DB::beginTransaction();
        
                $product = Product::findOrFail($id); 
                // Update user data
                $product->product_name = $request->product_name;
                $product->product_description = $request->product_description;
                $product->product_image_id = $request->product_image_id ?? $product->product_image_id;
                $product->product_type = $request->product_type;
                $product->display_box = $request->display_box;
                $product->status = $request->status ?? $product->status;
                $product->save();

                $product->product_field()->delete();


                foreach ($request->fields as $field) {
                    ProductField::create([
                        'product_id' => $product->id,
                        'field' => $field
                    ]);
                }

    
                DB::commit();
        
                return response()->json(['message' => 'Product updated successfully', 'record' => new ProductCollection($product)], 200);
            } catch (ModelNotFoundException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Product not found'], 404);
            } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
    
        /**
         * Remove the specified resource from storage.
         */
        public function destroy($id)
        {
            try {
                $ids = explode(",",$id);
                if (!empty($ids)) {
                    // Fetch the product  with the specified IDs
                    $products = Product::whereIn('id', $ids)->get();
                    
                    // Delete the fetched product groups
                    foreach ($products as $product) {
                        $product->delete();
                        $product->product_field()->delete();
                    }
                }
                    return response()->json(['message' => 'Product and associated entries deleted successfully'], 200);
            } catch (ModelNotFoundException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Product not found'], 404);
            } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }

        public function activeProduct(){
            try{
                $product = Product::where('status',1)->get();
                return response()->json(['message' => 'Product Details Fetached successfully', 'record' => ProductCollection::collection($product)], 200);
            }catch (\Exception $e) {
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
}
