<?php

namespace App\Http\Controllers;

use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\DB;
use App\Http\Resources\Role as RoleCollection;

class RoleController extends Controller
{
    public function index(Request $request)
        {
            try {
                $limit = $request->query('limit', 10);

                $query = Role::query();
                // Get all parameters from the request
                $parameters = $request->all();

            // Loop through each parameter and apply filters to the query
                foreach ($parameters as $key => $value) {
                    // Make sure the key is not a reserved Laravel parameter
                    if ($key !== 'page' && $key !== 'limit') {
                        if (in_array($key, ['role_name'])) {
                            // Apply search functionality using 'like' for name, email, and phone
                            $query->where($key, 'like', "%$value%");
                        } 
                        else {
                            // Apply exact match for other fields
                            $query->where($key, $value);
                        }
                    }
                }

                $roles = $query->orderBy('updated_at', 'desc')->paginate($limit);
                $roleCollection = RoleCollection::collection($roles->items());

                $responseData = [
                    'records' => $roleCollection,
                    'pagination' => [
                        'current_page' => $roles->currentPage(),
                        'last_page' => $roles->lastPage(),
                        'per_page' => $roles->perPage(),
                        'total' => $roles->total(),
                    ],
                ];
                return response()->json($responseData);
                
            } catch (QueryException $e) {
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
    
    
        /**
         * Store a newly created resource in storage.
         */
        public function store(Request $request)
        {
            try {
                $validated = $request->validate([
                    'role_name' => 'required|string|max:255',
                    'permission' => 'required|array', // Validate that permissions are an array
                ]);
    
                DB::beginTransaction();
                $role = Role::create([
                    'role_name' => $validated['role_name'], // Use validated name
                    'permission' => json_encode($validated['permission']),
                    'status' => $request->status ?? true, // Default to true if status is not provided.
                ]);
     
                DB::commit();
                return response()->json(['message' => 'Role created successfully', 'record' => new RoleCollection($role)], 201);
            } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
    
        /**
         * Display the specified resource.
         */
        public function show($id)
        {
            try{
                $role = Role::findOrFail($id);
                return response()->json(['message' => 'Role Details Fetached successfully', 'record' => new RoleCollection($role)], 200);
            }catch (\Exception $e) {
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
            
        }
    
    
        /**
         * Update the specified resource in storage.
         */
        public function update(Request $request, $id)
        {
            try {
                $validatedData = $request->validate([
                    'role_name' => 'required|string|max:100',
                ]);
    
                DB::beginTransaction();
        
                $role = Role::findOrFail($id); 
                // Update user data
                $role->role_name = $request->role_name;
                $role->permission = $request->permission;
                $role->status = $request->status ?? $role->status;
                $role->save();

                DB::commit();
                return response()->json(['message' => 'Role updated successfully', 'record' => new RoleCollection($role)], 200);
            } catch (ModelNotFoundException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Role not found'], 404);
            } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
    
        /**
         * Remove the specified resource from storage.
         */
        public function destroy($id)
        {
            try {
                $ids = explode(",",$id);
                if (!empty($ids)) {
                    // Fetch the role  with the specified IDs
                    $roles = Role::whereIn('id', $ids)->get();
                    
                    // Delete the fetched role groups
                    foreach ($roles as $role) {
                        $role->delete();
                    }
                }
                    return response()->json(['message' => 'Role and associated entries deleted successfully'], 200);
            } catch (ModelNotFoundException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Role not found'], 404);
            } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }

        public function activeRole(Request $request)
        {
            try {

                $roles = Role::where('status',1)->get();
                return response()->json(['records' => RoleCollection::collection($roles)]);
                
           } catch (QueryException $e) {
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
}
